import React, { PureComponent } from 'react';
import { Checkbox } from 'antd';

import './app-checkbox-group.less';

/**
 * 输入属性
 *
 * @export
 * @interface CheckBoxGroupProps
 */
export interface CheckBoxGroupProps {
    /**
     * 传入表单项值
     *
     * @type {any}
     * @memberof CheckBoxGroupProps
     */
    value?: any;

    /**
     * 代码表标识
     * 
     * @type {string}
     * @memberof CheckBoxGroupProps
     */
    tag?: string;

    /**
     * 是否禁用
     * 
     *  @type {boolean}
     * @memberof CheckBoxGroupProps
     */
    disabled: boolean;

    /**
     * 代码表
     * 
     *  @type {Object}
     * @memberof CheckBoxGroupProps
     */
    codelist?: Object;

    /**
     * 当前模式
     * 
     *  @type {string}
     * @memberof CheckBoxGroupProps
     */
    mode?: string;

    /**
     * 分割符
     * 
     *  @type {string}
     * @memberof CheckBoxGroupProps
     */
    separator?: string;

    /**
     * 值变化事件
     *
     * @memberof CheckBoxGroupProps
     */
    onChange: (value: string) => void;
}

/**
 * AppRadioGroup
 *
 * @export
 * @class AppRadioGroup
 * @extends {PureComponent}
 */
export default class AppCheckBoxGroup extends PureComponent<CheckBoxGroupProps> {

    /**
     * 当前代码表
     * 
     * @memberof AppCheckBoxGroup
     */
    public curCodeList: Array<any> = [];

    /**
     * 绑定值
     */
    public currentValue: any = undefined;

    /**
     * 是否禁用
     */
    public curDisabled: boolean = false;

    /**
     * 当前模式
     */
    public curMode?: string;

    /**
     * 当前分隔符
     */
    public curSeparator?: string;

    constructor(props) {
        super(props);
        if (props.tag && props.codelist) {
            this.curCodeList = props.codelist[props.tag];
        }
        if (props.separator) {
            this.curSeparator = props.separator;
        } else {
            this.curSeparator = ";";
        }
        if (props.mode) {
            this.curMode = props.mode;
        } else {
            this.curMode = "str";
        }
        if (props.value) {
            this.initValve(props.value);
        } else {
            this.currentValue = [];
        }
        if (props.disabled) {
            this.curDisabled = true;
        } else {
            this.curDisabled = false;
        }
    }

    /**
     * 在组件接收到一个新的 prop (或更新后)时被调用
     *
     * @memberof AppCheckBoxGroup
     */
    public componentWillReceiveProps(nextProps: Readonly<CheckBoxGroupProps>, nextContext: any) {
        if (nextProps.value) {
            this.initValve(nextProps.value);
        } else {
            this.currentValue = [];
        }
        if (nextProps.disabled) {
            this.curDisabled = true;
        } else {
            this.curDisabled = false;
        }
    }

    /**
     * 初始化value值
     */
    public initValve(value: any) {
        if (value) {
            if (Object.is(this.curMode, "num") && this.curCodeList) {
                let selectsArray: Array<any> = [];
                let num = parseInt(value, 10);
                this.curCodeList.forEach(item => {
                    if ((num & item.id) == item.id) {
                        selectsArray.push(item.id);
                    }
                });
                this.currentValue = selectsArray;
            } else if (Object.is(this.curMode, "str")) {
                if (value !== "") {
                    this.currentValue = value.split(this.curSeparator);
                } else {
                    this.currentValue = [];
                }
            }
        } else {
            this.currentValue = [];
        }
    }

    /**
     * 绘制内容
     *
     * @returns
     * @memberof AppCheckBoxGroup
     */
    public render() {
        return (<Checkbox.Group value={this.currentValue} style={{ width: '100%' }} disabled={this.curDisabled} onChange={($event) => { this.handleChange($event) }}>
            {this.curCodeList.map((item) => {
                return (<Checkbox key={item.id} value={item.id}>{item.text}</Checkbox>)
            })}
        </Checkbox.Group>
        );
    }

    /**
     * 值变化事件
     * 
     * @param val
     * @memberof AppCheckBoxGroup 
     */
    public handleChange(val) {
        let value:any;
        if (Object.is(this.curMode, "num")) {
            let temp = 0;
            val.forEach(item => {
                temp = temp | parseInt(item, 10);
            });
            value = temp;
        } else if (Object.is(this.curMode, "str")) {
            let _datas:Array<any> = [];
            this.curCodeList.forEach(item => {
                const index = val.findIndex(_key => Object.is(item.id, _key));
                if (index === -1) {
                    return;
                }
                _datas.push(item.id);
            });
            value = _datas.join(this.curSeparator);
        }
        this.props.onChange(value);
    }

    /**
     * 变更监测
     *
     * @protected
     * @memberof AppCheckBoxGroup
     */
    protected tick(): void {
        this.setState({
            date: new Date()
        });
    }

}