import React, { PureComponent } from 'react';
import { Upload, Icon, Modal } from 'antd';

import './app-image-upload.less';

/**
 * 输入属性
 *
 * @export
 * @interface ImageUploadProps
 */
export interface ImageUploadProps {
    /**
     * 传入表单项值
     *
     * @type {any}
     * @memberof ImageUploadProps
     */
    value: any;

    /**
     * 是否禁用
     * 
     *  @type {boolean}
     * @memberof ImageUploadProps
     */
    disabled?: boolean;

    /**
     * 是否多选
     */
    multiple?: boolean;

    /**
     * 上传参数
     */
    uploadparams?: string;

    /**
     * 导出参数
     */
    exportparams?: string;

    /**
     * 自定义参数
     */
    customparams?: Object;

    /**
     * 值变化事件
     *
     * @memberof ImageUploadProps
     */
    onChange: (value: string) => void;
}

/**
 * BookTestEditView
 *
 * @export
 * @class BookTestEditView
 * @extends {PureComponent}
 */
export default class AppImageUpload extends PureComponent<ImageUploadProps> {

    /**
     * 文件集合
     */
    public currentValue: Array<any> = [];

    /**
     * 是否禁用
     */
    public curDisabled: boolean = false;

    /**
     * 是否显示预览
     */
    public previewVisible: boolean = false;

    /**
     * 是否多选
     */
    public isMultiple: boolean = false;

    /**
     * 预览图片地址
     */
    public previewImage: string = "";

    /**
     * 上传路径
     */
    public uploadUrl: string = "https://www.mocky.io/v2/5cc8019d300000980a055e76";

    constructor(props) {
        super(props);
        if (props.value) {
            this.currentValue = this.handleInputData(props.value);
        } else {
            this.currentValue = [];
        }
        if (props.disabled) {
            this.curDisabled = true;
        } else {
            this.curDisabled = false;
        }
        if (props.multiple) {
            this.isMultiple = true;
        } else {
            this.isMultiple = false;
        }
    }

    /**
     * 在组件接收到一个新的 prop (或更新后)时被调用
     *
     * @memberof AppImageUpload
     */
    public componentWillReceiveProps(nextProps: Readonly<ImageUploadProps>, nextContext: any) {
        if (nextProps.value) {
            this.currentValue = this.handleInputData(nextProps.value);
        } else {
            this.currentValue = [];
        }
        if (nextProps.disabled) {
            this.curDisabled = true;
        } else {
            this.curDisabled = false;
        }
    }

    /**
     * 获取文件Base64
     * @param file 
     */
    public getBase64(file: any) {
        return new Promise((resolve, reject) => {
            const reader = new FileReader();
            reader.readAsDataURL(file);
            reader.onload = () => resolve(reader.result);
            reader.onerror = error => reject(error);
        });
    }
    /**
     * 处理预览文件
     */
    handlePreview = (file: any) => {
        if (!file.url && !file.preview) {
            this.getBase64(file.originFileObj).then((res) => {
                file.preview = res;
                this.previewImage = file.url || file.preview;
                this.previewVisible = true;
            }).catch((error) => {
                console.warn("处理预览图片错误" + error);
            })
        } else {
            this.previewImage = file.url || file.preview;
            this.previewVisible = true;
        }
        this.tick();
    }

    /**
     * 处理取消
     */
    handleCancel = () => {
        this.previewVisible = false;
        this.tick();
    }

    /**
     * 最多可上传多少图片
     */
    public getMaxLength() {
        if (this.isMultiple) {
            return 3;
        } else {
            return 1;
        }
    }



    /**
     * 绘制内容
     *
     * @returns
     * @memberof AppImageUpload
     */
    public render() {
        return (
            <div className="clearfix">
                <Upload action={this.uploadUrl}
                    accept="image/*"
                    listType="picture-card"
                    disabled={this.curDisabled}
                    fileList={this.currentValue}
                    onPreview={this.handlePreview}
                    onChange={this.handleChange} >
                    {this.currentValue.length >= this.getMaxLength() ? null : <div>
                        <Icon type="plus" />
                        <div className="ant-upload-text">Upload</div>
                    </div>}
                </Upload>
                <Modal visible={this.previewVisible} footer={null} onCancel={this.handleCancel}>
                    <img alt="example" style={{ width: '100%' }} src={this.previewImage} />
                </Modal>
            </div>
        );
    }

    /**
     * 处理输入参数
     * @param fileList 
     */
    public handleInputData(fileList: any) {
        //todo
        return fileList;
    }

    /**
     * 处理输出参数
     * @param fileList 
     */
    public handleOutData(fileList: any) {
        //todo
        return fileList;
    }

    /**
     * 值变化事件
     * 
     * @param $event
     * @memberof AppImageUpload 
     */
    public handleChange = ({ fileList }) => {
        this.props.onChange(this.handleOutData(fileList));
        this.tick();
    }

    /**
     * 变更监测
     *
     * @protected
     * @memberof AppImageUpload
     */
    protected tick(): void {
        this.setState({
            date: new Date()
        });
    }

}